/*
 * guidance.c
 *
 *  Created on: Apr 8, 2015
 *      Author: Jordan
 */


#include "sys_common.h"
#include "pwm.h"
#include "config.h"
#include "guidance.h"
#include "navigation.h"
#include "control.h"
#include "stateflow.h"
#include "math.h"

//float32 wpt_lat[N_WAYPOINTS] = {28.054933,		28.055110, 		28.054883, 		28.054583};
//float32 wpt_lon[N_WAYPOINTS] = {-82.494932,		-82.495030, 	-82.495356, 	-82.495183};
//float32 wpt_alt[N_WAYPOINTS] = {1, 				2, 				2,				1};
float32 wpt_lat[N_WAYPOINTS] = {28.053545,		28.053816, 		28.053830, 		28.054070,		28.054264,		28.054271,		28.054485,		28.054485,		28.054264,		28.054070,		28.053899,		28.053830,		28.053545,		28.053569,		28.053830};
float32 wpt_lon[N_WAYPOINTS] = {-82.501365,		-82.501366, 	-82.501759, 	-82.501653,		-82.501500,		-82.501202,		-82.501202,		-82.501476,		-82.501500,		-82.501653,		-82.501824,		-82.502109,		-82.502109,		-82.501562,		-82.501562};
float32 wpt_alt[N_WAYPOINTS] = {5, 				9, 				9,				9,				9,				9,				9,				9,				9,				9,				9,				7,				5,				5,				-10};
uint16 wpt_counter = 0;

void reset_wpt_counter(){
	wpt_counter = 0;
}

uint16 get_wpt_counter(){
	return wpt_counter;
}

float32 get_dist_to_wpt(){
	return sqrtf(powf(get_wpt_x(wpt_lat[wpt_counter]) - get_NED_x(),2) + powf(get_wpt_y(wpt_lat[wpt_counter], wpt_lon[wpt_counter]) - get_NED_y(),2));
}

float32 get_course_to_wpt(){
	return atan2f(get_wpt_y(wpt_lat[wpt_counter], wpt_lon[wpt_counter]) - get_NED_y(), get_wpt_x(wpt_lat[wpt_counter]) - get_NED_x());
}

float32 get_ail_cmd(){
	float32 cmd =  get_PWM_width(1) - AIL_MID;
	cmd = (cmd > 0 ? cmd*AIL_SC_HIGH : cmd*AIL_SC_LOW) + 1.5;
	return cmd > 2 ? 2 : cmd < 1 ? 1 : cmd;
}

float32 get_ele_cmd(){
	float32 cmd =  get_PWM_width(2) - ELE_MID;
	cmd = (cmd > 0 ? cmd*ELE_SC_HIGH : cmd*ELE_SC_LOW) + 1.5;
	return cmd > 2 ? 2 : cmd < 1 ? 1 : cmd;
}

float32 get_rud_cmd(){
	float32 cmd =  get_PWM_width(4) - RUD_MID;
	cmd = (cmd > 0 ? cmd*RUD_SC_HIGH : cmd*RUD_SC_LOW) + 1.5;
	return cmd > 2 ? 2 : cmd < 1 ? 1 : cmd;
}

float32 get_thr_cmd(){
	float32 cmd =  get_PWM_width(3) - THR_MIN;
	cmd = cmd*THR_SC + 1;
	return cmd > 2 ? 2 : cmd < 1 ? 1 : cmd;
}

void guidance_step(){
	float32 ail = get_ail_cmd();
	float32 ele = get_ele_cmd();
	float32 rud = get_rud_cmd();
	float32 thr = get_thr_cmd();
	float32 aux = get_PWM_width(5);

	ail = fabs(ail - 1.5) > STICK_DEAD_ZONE ? ail : 1.5;
	ele = fabs(ele - 1.5) > STICK_DEAD_ZONE ? ele : 1.5;
	rud = fabs(rud - 1.5) > STICK_DEAD_ZONE ? rud : 1.5;

	ail = (ail - 1.5)*2;
	ele = (ele - 1.5)*2;
	rud = (rud - 1.5)*2;

	ail = ail > 0 ? pow(ail,EXPO_CURVE) : -pow(-ail,EXPO_CURVE);
	ele = ele > 0 ? pow(ele,EXPO_CURVE) : -pow(-ele,EXPO_CURVE);
	rud = rud > 0 ? pow(rud,EXPO_CURVE) : -pow(-rud,EXPO_CURVE);

	if(get_flight_mode() == FLIGHT_MODE_WAYPOINT){
		if(get_dist_to_wpt() < WPT_THRESH){
			wpt_counter = wpt_counter == N_WAYPOINTS - 1 ? 0 : wpt_counter + 1;
		}

		float32 alt_error = -wpt_alt[wpt_counter] - get_NED_z();
		alt_error = alt_error > GUIDANCE_ALT_ERR_LIM ? GUIDANCE_ALT_ERR_LIM : alt_error < -GUIDANCE_ALT_ERR_LIM ? -GUIDANCE_ALT_ERR_LIM : alt_error;
		set_vel_z_ref(GUIDANCE_ALT_GAIN*alt_error);

		float32 yaw_error_full = get_course_to_wpt() - get_yaw();
		yaw_error_full = yaw_error_full > PI ? yaw_error_full - 2*PI : yaw_error_full < -PI ? yaw_error_full + 2*PI : yaw_error_full;
		float32 yaw_error = yaw_error_full > GUIDANCE_YAW_ERR_LIM ? GUIDANCE_YAW_ERR_LIM : yaw_error_full < -GUIDANCE_YAW_ERR_LIM ? -GUIDANCE_YAW_ERR_LIM : yaw_error_full;
		set_yaw_rate_ref(GUIDANCE_YAW_GAIN*yaw_error);

		set_vel_y_ref(0);

		if(wpt_counter == N_WAYPOINTS - 1){
			set_vel_x_ref(0);
		} else {
			set_vel_x_ref(VEL_X_SC*(cosf(yaw_error_full)+1)/2);
		}

//		sci_printf("r: %5.2f, wpt: %u, vel: %5.2f\r\n", GUIDANCE_YAW_GAIN*yaw_error*180/PI, get_wpt_counter(), VEL_X_SC*(cosf(yaw_error_full)+1)/2);
	} else if(get_flight_mode() == FLIGHT_MODE_VELOCITY){
		set_vel_y_ref(ail);
		set_vel_x_ref(-ele);
		thr = fabs(thr - 1.5) > VEL_Z_DEAD_ZONE ? thr : 1.5;
		set_vel_z_ref(-2*(thr - 1.5));
		set_yaw_rate_ref(thr > 1 + CMD_THRESHOLD ? rud : 0);
	} else if(get_flight_mode() == FLIGHT_MODE_ANGLE){
		set_roll_ref(ail);
		set_pitch_ref(ele);
		set_thr_ref(thr - 1);
		set_yaw_rate_ref(thr > 1 + CMD_THRESHOLD ? rud : 0);
	} else {
		set_roll_rate_ref(ail);
		set_pitch_rate_ref(ele);
		set_thr_ref(thr - 1);
		set_yaw_rate_ref(thr > 1 + CMD_THRESHOLD ? rud : 0);
	}

	#if RECEIVER_DEBUG == 1
		sci_printf("Ail: %5.2f, Ele: %5.2f, Rud: %5.2f, Thr: %5.2f, Aux: %5.2f\r\n", ail, ele, rud, thr, aux);
//		sci_printf("Ail: %5.2f, Ele: %5.2f, Rud: %5.2f, Thr: %5.2f, Aux: %5.2f\r\n", get_PWM_width(1), get_PWM_width(2), get_PWM_width(4), get_PWM_width(3), aux);
	#endif
}
